;mastst2a.asm to develop MSSP communications with the LCD PIC.
;john waller 2002nov17.

#DEFINE BANK0   BCF $03,5   ;STATUS bit 5
#DEFINE BANK1   BSF $03,5   ;STATUS bit 5

INDF:    .EQU $00	;Indirect address, bank 0, 1, 2, 3.
TMRO:	 .EQU $01	;Timer 0 value, bank 0, 3.
OPTION:  .EQU $01	;Option register, bank 1, 3.
PCL:     .EQU $02	;Program counter low byte, bank 0, 1, 2, 3.
STATUS:  .EQU $03	;Status flags, bank 0, 1, 2, 3.
FSR:     .EQU $04	;Indirect address pointer, bank 0, 1, 2, 3.
PORTA:   .EQU $05	;Port A, bank 0.
TRISA:   .EQU $05	;Port A direction register, bank 1.
PORTB:   .EQU $06	;Port B, bank 0, 2.
TRISB:   .EQU $06	;Port B direction register, bank 1, 3.
PORTC:   .EQU $07	;Port C, bank 0.
TRISC:   .EQU $07	;Port C direction register, bank 1.
PORTD:   .EQU $08	;Port D, bank 0.
TRISD:   .EQU $08	;Port D direction register, bank 1.
PORTE:   .EQU $09	;Port E, bank 0.
TRISE:   .EQU $09	;Port E direction register, bank 1.
PCLATH:	 .EQU $0A	;Program counter high byte setting register, bank 0, 1, 2, 3.
INTCON:  .EQU $0B	;Interrupt control, bank 0, 1, 2, 3.
PIR1:	 .EQU $0C	;Peripheral interrupt flags, bank 0.
PIE1:    .EQU $0C	;Peripheral interrupt enable, bank 1.
PIR2:    .EQU $0D	;Peripheral interrupt flags, bank 0.
PIE2:    .EQU $0D	;Peripheral interrupt enable, bank 1.
TMR1L:	 .EQU $0E	;Timer 1 low byte, bank 0.
TMR1H:	 .EQU $0F	;Timer 1 high byte, bank 0.
T1CON:	 .EQU $10	;Timer 1 control, bank 0.
SSPCON2: .EQU $11	;SSP configuration, bank 1.
SSPBUF:	 .EQU $13	;SSP address and data buffer, bank 0.
SSPADD:  .EQU $13	;SSP master baud rate/slave address, bank 1.
SSPCON:  .EQU $14	;SSP configuration, bank 0.
SSPSTAT: .EQU $14	;SSP status, bank 1.
ADCON1:	 .EQU $1F	;Port A A/D control register, bank 1.

;........GENERAL PURPOSE REGISTERS

MBYTCTR: .EQU $20	;SSP data byte count, bank 0.
MBYTNUM: .EQU $21	;SSP number of data bytes to exchange, bank 0.
MINTCTR: .EQU $22	;SSP interrupt count, bank 0.
SYSTEM:  .EQU $23	;System parameters: bit 0 MSSP; 0 = send, 1 = read:
			;bit 1 MSSP 0 message not sent 1 sent:
			;others not yet assigned.
TMR1HS:	 .EQU $24	;Timer 1 high byte stored, bank 0.
ACTIONC: .EQU $25	;Action counter updated by button press, bank 0.
PORTAS:  .EQU $26	;Port A value stored, bank 0.

			;Arrays accessible with IRP cleared.
MREABYT: .EQU $A0	;MSSP bytes read from slave to A7h.
MSENBYT: .EQU $A8	;MSSP bytes sent to slave to AFh.

			;Registers to save working, FSR, STATUS, and PCLATH at interrupt.
			;Must be visible across all banks, 70H..7FH.
FSRINT:	.EQU $7C	;For FSR register, alias, FCH, 17CH, & 1FCH.
PCLINT:	.EQU $7D	;For PCLATH register, alias, FDH, 17DH, & 1FDH.
STAINT:	.EQU $7E	;For STATUS register, alias, FEH, 17EH, & 1FEH.
WRKINT:	.EQU $7F	;For working register, alias FFH, 17FH, & 1FFH.

;........CONSTANT VALUES

ERRINT:	.EQU $20	;Tag for an interrupt error.
ERRSAD:	.EQU $40	;Tag for a master address error.
ERRSAK:	.EQU $60	;Tag for a failure of slave to acknowledge.
ERRSDT:	.EQU $80	;Tag for a master data load error.
SLVADD: .EQU $10	;Slave address; upper 7 bits; equate must be even;
			;do not use F8h, FAh, FCh, FEh, or zero.
			;16F877 manual says only 112 valid addresses available;
			;don't know what the other invalid ones are.

;........BIT VALUES

ACKDT:  .EQU 5		;SSP master acknowledge data.
ACKEN:  .EQU 4		;SSP master 
ACKSTAT:.EQU 6		;SSP master acknowledge status.
BCLIE   .EQU 3		;Bus collision interrupt enable.
BCLIF:  .EQU 3		;Bus collision interrupt flag.
BF:     .EQU 0		;SSP buffer full.
C:      .EQU 0          ;Carry status.
CKE:    .EQU 6		;Clock edge select.
CKP:	.EQU 4		;Hold/enable SSP clock pulse by slave.
DIR:	.EQU 0		;MSSP direction; 0 send, 1 read.
DNA     .EQU 5		;Last byte received or transmitted was data or address.
DONE:	.EQU 1		;MSSP message executed; 0 not yet, 1 done.
F:      .EQU 1          ;Select file register.
GCEN:   .EQU 7		;SSP general call enable.
GIE:    .EQU 7		;General interrupt enable.
P:      .EQU 4		;Stop bit.
PEIE:   .EQU 6		;Peripheral interrupt enable.
PEN:    .EQU 2		;SSP master stop enable.
RCEN:   .EQU 3		;SSP receive enable.
RNW:    .EQU 2		;Read/write information following the last address match.
RP0:    .EQU 5          ;STATUS bank control bit.
RP1:    .EQU 6          ;STATUS bank control bit.
RSEN:	.EQU 1		;SSP master repeat start enable.
SEN:    .EQU 0		;SSP master start enable.
SMP:    .EQU 7		;Slew rate control.
SSPIE:  .EQU 3		;SSP interrupt enable.
SSPIF:  .EQU 3		;SSP interrupt flag.
SSPEN:  .EQU 5		;Configure SDA and SCL pins for SSP.
SSPOV:  .EQU 6		;SSP receive overflow detect.
UA:     .EQU 1		;Update address; 10-bit address mode only.
W:      .EQU 0          ;Select working register.
WCOL:   .EQU 7		;SSP write collision detect.
Z:      .EQU 2          ;Zero status.

;..........

        .ORG $0004
	goto INTERRUPT
        .ORG $0005

	goto INITIALISE	;jump over sub-page 0 tables to INITIALISE.

		;page 0, sub-page 0 tables: make sure they don't overrun.
		
;..........

BACKGROUND:	;Main part of program in background; entered from INITIALISE.
		;Called from none; calls none.
BACK1:	btfsc PORTA,4		;Loop until
	goto BACK1		;switch closed.
	clrf MINTCTR		;Clear MSSP interrupt count.
	clrf MBYTCTR		;Clear MSSP byte count.
	BANK1			;
	bsf SSPCON2,SEN		;Start transmission.
	BANK0			;
BACK2:	goto BACK2		;Pause.
	goto BACKGROUND		;Loop back.

;..........

BEGCRITSECTION:	;Ensures GIE is cleared before executing the code which succeeds
		;this call.
	bcf INTCON,GIE		;Clear all interrupts.
	btfsc INTCON,GIE	;Skip if clearing succeeded.
	goto BEGCRITSECTION	;Otherwise keep trying.
	return			;
;..........

CHECKANDSTOP:	;Writes selected registers to ports and pauses.
		;Called from ERRORINT, ERRORSACKT, ERRORSADDT, ERRORSDATT; calls, none.
	BANK0			;
	andlw %11100000		;Write high bits in W to port C
	movwf PORTC		;to indicate where in program stop occurred.
	bsf PORTE,2		;Signal program has reached here.
	btfss PIR1,SSPIF	;Show SSP
	goto CHAST1		;interrupt
	bsf PORTC,2		;bit
	goto CHAST2		;on
CHAST1:	bcf PORTC,2		;port C,2.
CHAST2:	BANK1			;Write
	movf SSPSTAT,W		;register
	BANK0			;contents
	movwf PORTB		;to port.
	movf SSPCON,W		;Write register
	movwf PORTD		;contents to port.
CHAST3:	goto CHAST3		;Loop indefinitely.

;..........

DELAY0PT2S:	;Delays approximately 0.2 second using timer 1.
		;Called from ...............; calls none.
	movf TMR1H,W		;Store current value of
	movwf TMR1HS		;timer 1 high byte.
DE0P21:	movf TMR1H,W		;Skip if timer
	subwf TMR1HS,W		;high byte
	sublw 1			;is one less than
	btfss STATUS,Z		;stored value.
	goto DE0P21		;Otherwise continue to wait.
	return			;

;..........

DISPACTIONCTR:	;Displays the action counter, showing lower three bits on port
		;A,0 ,1, 2.
	movf ACTIONC,W		;Mask in lower three bits
	andlw %00000111		;of action counter
	movwf PORTAS		;and store result.
	movf PORTA,W		;Mask in upper
	andlw %00111000		;three bits of port A,
	addwf PORTAS,W		;add in lower three bits and
	movwf PORTA		;write to port A.
	movf ACTIONC,W		;Carry out action
	call SWITCHCASE8	;designated by value.
	return

;..........

ENDCRITSECTION	;Re-eanables all interrupts following a critical section.
	bsf INTCON,GIE		;Enable all interrupts.
	return			;
	
;..........

ERRORINT:	;Sets tag for an interrupt error and halts execution.
		;Called from INTERRUPT; calls CHECKANDSTOP.
	movlw ERRINT		;Set interrupt error tag, write SSPSTAT and SSPCON2 to
	call CHECKANDSTOP	;ports B and D, respectively, and halt.

;..........

ERRORSACKT:	;Sets tag for a failure of slave to acknowledge with ACK.
		;Called from HANDSSPACKT; calls CHECKANDSTOP.
	movlw ERRSAK		;Set slave acknowledge failure tag, write SSPSTAT
				;and SSPCON to ports B and D,
	call CHECKANDSTOP	;respectively, and halt.

;..........

ERRORSADDT:	;Sets tag for a master transmit address error and halts execution.
		;Called from HANDSSPADDT; calls CHECKANDSTOP.
	movlw ERRSAD		;Set master transmit address error tag, write SSPSTAT
				;and SSPCON to ports B and D,
	call CHECKANDSTOP	;respectively, and halt.

;..........

ERRORSDATT:	;Sets tag for a master transmit data error and halts execution.
		;Called from HANDSSPDATT; calls CHECKANDSTOP.
	movlw ERRSDT		;Set master transmit data error tag, write SSPSTAT
				;and SSPCON to ports B and D,
	call CHECKANDSTOP	;respectively, and halt.

;..........

HANDSSPACKT:	;Handles an acknowledgment from slave and halts execution if not ACK.
		;Called from INTERRUPT; calls ERRORSACKT.
	BANK1			;
	btfsc SSPCON2,ACKSTAT	;Skip if ACK was received,
	call ERRORSACKT		;otherwise report error and halt.
	BANK0			;
	return			;
	
;..........

HANDSSPADDT:	;Handles a start complete interrupt by sending a slave address and
		;checking for a write collision error.
		;Called from INTERRUPT; calls ERRORSADDT.
	movf SYSTEM,W		;Add system direction bit
	andlw %00000001		;to slave address and
	addlw SLVADD		;write slave address to SSP buffer
	movwf SSPBUF		;to start transmission.
	btfsc SSPCON,WCOL	;Skip if no write collision,
	call ERRORSADDT		;otherwise report error and halt.
	incf MINTCTR,F		;Increment interrupt count.
	return			;
	
;..........

HANDSSPDATT:	;Handles a master send data interrupt by filling buffer and checking
		;for a write collision error.
		;Called from INTERRUPT; calls ERRORSDATT.
	movlw MSENBYT		;Point to send byte
	addwf MBYTCTR,W		;array current
	movwf FSR		;storage location.
	movf INDF,W		;Put contents therefrom
	movwf SSPBUF		;into buffer.
	btfsc SSPCON,WCOL	;Skip if no write collision,
	call ERRORSDATT		;otherwise report error and halt.
	incf MBYTCTR,F		;Increment byte and
	incf MINTCTR,F		;interrupt counters.
	return			;
	
;..........

HANDSSPRBFS:	;Handles a read byte from slave interrupt.
		;Called from INTERRUPT; calls none.
	movlw MREABYT		;Point to read byte
	addwf MBYTCTR,W		;array current
	movwf FSR		;storage location.	
	movf SSPBUF,W		;Put buffer
	movwf INDF		;contents there.
	incf MBYTCTR,F		;Increment byte and
	incf MINTCTR,F		;interrupt counters.
	return			;
	
;..........

HANDSSPSATS:	;Handles interrupt to send ACK to slave.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bcf SSPCON2,ACKDT	;Send ACK
	bsf SSPCON2,ACKEN	;to slave.
	BANK0			;
	return			;
	
;..........

HANDSSPSNTS:	;Handles interrupt to send NACK to slave.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bsf SSPCON2,ACKDT	;Send NACK
	bsf SSPCON2,ACKEN	;to slave.
	BANK0			;
	return			;
	
;..........

HANDSSPSRBF:	;Handles interrupt where read byte flag is to be set.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bsf SSPCON2,RCEN	;Set flag.
	BANK0			;
	incf MINTCTR,F		;Increment interrupt count.
	return			;
	
;..........

HANDSSPSTOP:	;Handles an interrupt where 'stop' is to be initiated.
		;Called from INTERRUPT; calls none.
	BANK1			;
	bsf SSPCON2,PEN		;Initiate 'stop'.
	BANK0			;
	incf MINTCTR,F		;Increment interrupt count. 
	return			;
	
;..........

INITI2CMSTR:	;Initialises the I2C master device.
		;Called from INITIALISE; calls none.
	movlw %00101000		;Bits 7, 6, flags; bit 5 enable SDA and SCL port pins;
	movwf SSPCON		;bit 4, unused; bits 3..0, I2C master mode.
	BANK1			;
	movlw %10000000		;Bit 7, slew rate control disabled; bit 6, comply with
	movwf SSPSTAT		;I2C protocol; bits 5..0, flags.
	clrf SSPCON2		;All bits flags.
	movlw 24		;Set baud rate to 100 kHz 
	movwf SSPADD		;for 10 MHz oscillator.
	bsf PIE1,SSPIE		;Enable SSP interrupt.
	BANK0			;
	bsf INTCON,PEIE		;Enable peripheral interrupts.
	bsf INTCON,GIE		;Enable general interrupts.
	return			;
		
;..........

INITIALISE:	;Initialising ports and others functions. Entered at startup.
		;Called from none; calls INITI2CMSTR.
	clrf PCLATH		;Page and sub-page 0.
	clrf STATUS		;Bank 0 and clear all flags.
				;Clear all gpr locations in banks 0 and 1.
	movlw $20		;First gpr address in bank 0
	movwf FSR		;to indirect address register.
INITB0:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss FSR,7		;Skip if end of bank 0.
	goto INITB0		;
	movlw $A0		;First gpr address in bank 1
	movwf FSR		;to indirect address register.
INITB1:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss STATUS,Z		;Skip if end of bank 1.
	goto INITB1		;
				;Set up ports and timers.
	clrf PORTA		;Clear port.
        clrf PORTB		;Clear port.
	clrf PORTC		;Clear port.
        clrf PORTD		;Clear port.
	clrf PORTE		;Clear port.
        BANK1			;
	movlw %00000110		;Port A as digital
	movwf ADCON1		;input/output.
	movlw %00110000		;Port A0..A3 outputs, A4, A5 inputs.
        movwf TRISA     	;as input.
        clrf TRISB      	;Port B0-B7 as output.
	movlw %00011000		;Port C all bits output, except
	movwf TRISC		;3 and 4 must be input for I2C.
	movlw %00000000		;Port D bits all
	movwf TRISD		;outputs.
	movlw %00000000		;Port E all bits as output,
	movwf TRISE		;disable slave port.
        BANK0			;
        clrf INTCON		;No interrupts or timers yet.
	movlw %00110001		;Set up timer 1; bit 0 enable, bit 1 Fosc/4, bit 2 X,
	movwf T1CON		;bit 3 not internal oscillator, bits 4/5 prescale 8,
				;bits 6/7 not implemented.
        call INITI2CMSTR	;Initialise master SSP.
        movlw 4			;Set number of data bytes
        movwf MBYTNUM		;to four.
        call SETDUMBYT2SEND	;Seeds the array to send data to slave with dummy data.
        call TESTACTIONCTR	;Test actions; loops in this routine continuously.
        goto BACKGROUND		;
        
	;...........

INTERRUPT:	;Interrupt routine goes here.
		;Called from none; calls ERRORINT, HANDSSPADDT, HANDSSPACKT,
		;HANDSSPDATT, HANDSSPSTOP.
				;Save registers and set up to execute interrupt.
	movwf WRKINT		;Save working register.
	swapf STATUS,W		;Save status
	movwf STAINT		;register, without affecting flags.
	movf PCLATH,W		;Save page
	movwf PCLINT		;register.
	movf FSR,W		;Save indirect
	movwf FSRINT		;address register.
	clrf STATUS		;Set bank, page,
	clrf PCLATH		;and sub-page 0.
	
				;Execute interrupt.
	btfss PIR1,SSPIF	;If not SSP interrupt, call error
	call ERRORINT		;routine and halt execution, otherwise service
	bcf PIR1,SSPIF		;interrupt, first clearing interrupt flag.
				;
	BANK1			;
	btfss SSPSTAT,P		;Skip if STOP in force.
	goto INTRT1		;
	BANK0			;
	clrf MBYTCTR		;Clear both
	clrf MINTCTR		;MSSP counters.
	goto INTEND		;
INTRT1:	BANK0			;
	btfss SYSTEM,DIR	;Skip if MSSP read.
	goto INTRT7		;
				;Read data from slave section.
	movf MINTCTR,F		;Skip if MSSP interrupt
	btfss STATUS,Z		;counter is zero.
	goto INTRT2		;
	call HANDSSPADDT	;Send address to slave.
	goto INTEND		;
				;
INTRT2:	movf MINTCTR,W		;Skip if
	sublw 1			;interrupt
	btfss STATUS,Z		;count is 1.
	goto INTRT3		;
	call HANDSSPACKT	;Check for acknowledgment.
	call HANDSSPSRBF	;Set read byte flag.
	goto INTEND		;
				;
INTRT3:	btfsc MINTCTR,0		;Skip if interrupt counter even.
	goto INTRT5		;
	call HANDSSPRBFS	;Read byte from slave.
	movf MBYTNUM,W		;Skip if enough
	subwf MBYTCTR,W		;bytes have
	btfss STATUS,Z		;been read.
	goto INTRT4		;
	call HANDSSPSNTS	;Send NACK to slave.
	goto INTEND		;
INTRT4:	call HANDSSPSATS	;Send ACK to slave.
	goto INTEND		;
INTRT5:	movf MBYTNUM,W		;Skip if enough
	subwf MBYTCTR,W		;bytes have
	btfss STATUS,Z		;been read.
	goto INTRT6		;
	call HANDSSPSTOP	;Initiate STOP.
	goto INTEND		;
INTRT6:	call HANDSSPSRBF	;Set read byte flag and
	;incf MINTCTR,F		;increment interrupt count.
	goto INTEND		;
				;Send data to slave section.
INTRT7:	movf MINTCTR,F		;Skip if MSSP interrupt
	btfss STATUS,Z		;counter is zero.
	goto INTRT8		;
	call HANDSSPADDT	;Send address to slave.
	goto INTEND		;
				;
INTRT8:	call HANDSSPACKT	;Check for acknowledgement.
	movf MBYTNUM,W		;Skip if enough
	subwf MBYTCTR,W		;bytes have
	btfss STATUS,Z		;been read.
	goto INTRT9		;
	call HANDSSPSTOP	;Initiate STOP.
	goto INTEND		;
				;
INTRT9:	call HANDSSPDATT	;Send a data byte.
	goto INTEND		;
	

INTEND:				;Restore registers and return.
	movf FSRINT,W		;Restore indirect
	movwf FSR		;address register.
	movf PCLINT,W		;Restore page
	movwf PCLATH		;register.
	swapf STAINT,W		;Restore status
	movwf STATUS		;register.
	swapf WRKINT,F		;Restore working
	swapf WRKINT,W		;register.
	retfie			;

;..........

SETDUMBYT2SEND:	;Sets up the send array with dummy data.
		;Called from .........., calls none.
	movlw MSENBYT		;
	movwf FSR		;
	movlw %00101000		;
	movwf INDF		;
	incf FSR,F		;
	movlw %10100110		;
	movwf INDF		;
	incf FSR,F		;
	movlw %11101010		;
	movwf INDF		;
	incf FSR,F		;
	movlw %01111010		;
	movwf INDF		;
	return			;
	
;..........

SWITCHCASE8:	;Takes first 3 bits of working register and does an 8-way switch-
		;case selection.
		;Called from ..............; calls ......
	andlw %00000111 	;Mask in 3 lower bits.
        btfss STATUS,Z  	;
        goto SWCAS1     	;
        			;Value 0.
        movlw MREABYT		;Point to array.
        ;call BEGCRITSECTION
        movwf FSR		;Get value and
        movf INDF,W		;display on
        ;call ENDCRITSECTION
        movwf PORTB		;port B.
        return     		;
SWCAS1: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS2     	;
        			;Value 1.
        movlw MREABYT		;Point to array.
        addlw 1			;Point to member of interest.
        ;call BEGCRITSECTION
        movwf FSR		;Get value and
        movf INDF,W		;display on
        ;call ENDCRITSECTION
        movwf PORTB		;port B.
        return     		;
SWCAS2: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS3     	;
        			;Value 2.
        movlw MREABYT		;Point to array.
        addlw 2			;Point to member of interest.
        ;call BEGCRITSECTION
        movwf FSR		;Get value and
        movf INDF,W		;display on
        ;call ENDCRITSECTION
        movwf PORTB		;port B.
        return     		;
SWCAS3: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS4     	;
        			;Value 3.
        movlw MREABYT		;Point to array.
        addlw 3			;Point to member of interest.
        ;call BEGCRITSECTION
        movwf FSR		;Get value and
        movf INDF,W		;display on
        ;call ENDCRITSECTION
        movwf PORTB		;port B.
        return     		;
SWCAS4: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS5     	;
        			;Value 4.
        btfsc SYSTEM,DONE	;Skip if message not already executed.
        return			;
        btfss PORTA,5		;If port A,5
        goto SWCS41		;is clear then
        bsf SYSTEM,DIR		;master sends.
        goto SWCS42		;otherwise
SWCS41: bcf SYSTEM,DIR		;master reads.
SWCS42:	BANK1			;
	bsf SSPCON2,SEN		;Start message.
	BANK0			;
	bsf SYSTEM,DONE		;Set message executed flag.
        return     		;
SWCAS5: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS6     	;
        			;Value 5.
        bcf SYSTEM,DONE		;Clear message executed flag.
        return     		;
SWCAS6: addlw $0FF      	;
        btfss STATUS,Z  	;
        goto SWCAS7     	;
        			;Value 6.
        return     		;
        			;Value 7 and default.
SWCAS7: return             	;



;..........

TESTACTIONCTR:	;Tests the action counter.
	call DISPACTIONCTR	;Display low bits on port A,0,1,2.
	call WAITFORSWPRESS	;Increment
	incf ACTIONC,F		;counter.
	goto TESTACTIONCTR	;

;..........

WAITFORSWPRESS:	;Waits for a momentary-action switch to pull down port A,4
		;and release it, with 0.2 second de-bounce delays.
	btfsc PORTA,4		;Wait for button to
	goto WAITFORSWPRESS	;pull port pin down.
	call DELAY0PT2S		;De-bounce delay.
WTFSP1:	btfss PORTA,4		;Wait for button to
	goto WTFSP1		;release port pin.
	call DELAY0PT2S		;De-bounce delay.
	return			;
	
;..........

        .END
